<?php
/**
 * @package Stripe Button System plugin
 * @author Riccardo Zorn code@fasterjoomla.com
 * @copyright (C) 2018 - 2022 https://www.fasterjoomla.com
 * @license GNU/GPL v2 or greater http://www.gnu.org/licenses/gpl-2.0.html
 */

defined( '_JEXEC' ) or die( 'Restricted access' );

jimport('joomla.plugin.plugin' );
jimport('joomla.filesystem.file');
jimport('joomla.application.application' );
use Stripe\Stripe as Stripe;

class plgSystemStripe_button extends JPlugin
{
	private $stripePlugin;
	private $stripeParams;
	private $scriptLibrary = 'https://js.stripe.com/v3/'; // @NOTE also in payments/stripe.php
	private $buttonIdCount = 0;	

	function __construct(& $subject, $config)
	{
		parent::__construct($subject, $config);
		$this->loadLanguage();
		ini_set('display_errors', 0);
		
		$this->stripePlugin = JPluginHelper::getPlugin('payment','stripe');
		if (!$this->stripePlugin) {
			if (JPATH_BASE == JPATH_ADMINISTRATOR) {
				JError::raiseWarning(521,JText::_("STRIPE_PAYMENT_NOT_AVAILABLE"));
			} 
			return true;
		}
		$this->stripeParams = new JRegistry($this->stripePlugin->params);
	}


	/**
	 * place the stripe payment button in place of
	 * {stripe-button amount="12.25" currency="EUR" description="buy you a beer"}
	 */
	public function onAfterRender() {
		if (!$this->stripePlugin) { return; }
		if (JPATH_BASE == JPATH_ADMINISTRATOR) return;
		
		$appWeb = JFactory::getApplication(); //new JApplicationWeb
		$action = $appWeb->input->get('action');
		if ($action=='stripepayment_webhook') {
			
		} else {
			if(version_compare(JVERSION,'3.0.0') < 0){
				$body = JResponse::getBody();
			} else {
				$body = $appWeb->getBody();
			}

			
			// joomla 2.5: JResponse::getBody()
			// joomla 3+: $appWeb->getBody();
			$expression = '/(\{stripe-button[^\}]*\})/i';
			$body = preg_replace_callback($expression, array(&$this,  'buildButtonCallback'), $body);
			if(version_compare(JVERSION,'3.0.0') < 0){
				JResponse::setBody($body);
			} else {
				$appWeb->setBody($body);
			}			
		}
	}
	/**
	 * Handle special urls (endpoints for stripe)
	 *
	 * @return void
	 */
	public function onAfterRoute() {
		if (!$this->stripePlugin) { return; }
		
		$input = JFactory::getApplication()->input;
		$action = $input->get('action');
		
		$adminActions = [
			'stripepayment_config_toolbar',
			'stripepayment_setup_delete_webhooks',
			'stripepayment_setup_list_webhooks',
			'stripepayment_setup_create_webhooks',
			'stripepayment_webhook',
		];		
		$userActions = [
			'stripebutton_render',
			'stripepayment_confirm',
			'stripepayment_webhook',  
		];	
		
		if (empty($input->get('task')) || !empty($action)) {
			$this->loadStripeLibrary();
		}			

		if (empty($action)) return;
		
		if (JPATH_BASE == JPATH_SITE &&  !in_array($action, $userActions)) {
			return;
		} 
		
		if (JPATH_BASE == JPATH_ADMINISTRATOR && !in_array($action , $adminActions)) {
			return;
		} 
		
		JPluginHelper::importPlugin( 'payment' , 'stripe');
		$dispatcher = JEventDispatcher::getInstance($this->stripePlugin);
		switch ($action) {
			case 'stripebutton_render':
				$this->handleRenderButton();
				die();
				break;
			case 'stripepayment_confirm':
				// this is the step after payment is confirmed by Checkout.
				$this->handlePaymentConfirmComponent();
				break;
			case 'stripepayment_webhook':
				// before the _confirm call returns, the webhook is invoked with the result
				$dispatcher->trigger('handlePaymentWebhookStripe', []);
				die();
				break;					
				
			case 'stripepayment_setup_create_webhooks': 
				// admin: create webhook in the current sandbox setting
				if (plgPaymentStripeHelper::isAdmin('{}')) {
					JEventDispatcher::getInstance($this->stripePlugin)->trigger('handleSetupWebhooksStripe', []);
				} 
				die();
				break;
		
			case 'stripepayment_setup_list_webhooks': 
				// admin: list all webhooks for the current sandbox setting
				if (plgPaymentStripeHelper::isAdmin('{}')) {
					JEventDispatcher::getInstance($this->stripePlugin)->trigger('handleListWebhooksStripe', []);
				}
				die();
				break;

			case 'stripepayment_setup_delete_webhooks':
				if (plgPaymentStripeHelper::isAdmin('{}')) {
					$this->handleDeleteWebhooks();
				}
				die();
				break;				

			case 'stripepayment_config_toolbar':
				if (plgPaymentStripeHelper::isAdmin('{}')) {
					$this->getToolbarData();
				}
				die();
				break;
		}
	}

	public function handleDeleteWebhooks() {
		//$this->stripePlugin = JPluginHelper::importPlugin( 'payment' , 'stripe');
		JEventDispatcher::getInstance($this->stripePlugin)->trigger('handleDeleteWebhooksStripe', []);
	}


	/**
	 * The stripe javascript library should be added to most site pages, to improve fraud detection.
	 * But the site admin can disable it on some pages through configuration.
	 * This event is invoked in the AfterRender event
	 *
	 * @return void
	 */
	public function loadStripeLibrary() {
		if (!defined('STRIPE_JS_LOADED')){
			$input = JFactory::getApplication()->input;
			$pageFilter = $this->stripeParams->get('loadStripeLibraryPagesFilter');
			$componentFilter = $this->stripeParams->get('loadStripeLibraryComponentsFilter');
			$pf = array();
			if (!empty($pageFilter)) {
				if (is_array($pageFilter)) {
					$pf = $pageFilter;
				} else if (strpos($pageFilter, ',')) {
					$pf = $pageFilter.splice(',');
				} else {
					$pf = $pageFilter.splice("\n");
				}
				
				if (in_array($input->get('Itemid'), $pf)) {
					// error_log('Stripe: NOT loading js on this page: ' . $pageFilter . ', ' . $input->get('Itemid'));
					return; 
				}
			}
			if (!empty($componentFilter)) {
				$cf = $componentFilter.splice(',');
				if (in_array($input->get('option'), $cf)) { 
					// error_log('Stripe: NOT loading js on this component: ' . $componentFilter . ', ' . $input->get('option'));
					return;
				}
			}
			$document = JFactory::getDocument();
			$document->addScript($this->scriptLibrary);
		  define('STRIPE_JS_LOADED', '1');
		} else {
			return;
		}
		
		
	}

	public function handleRenderButton() {
		$this->loadStripeLibrary();
		$input = JFactory::getApplication()->input;
		$description = $input->get('description', '', 'string');
		$amount = $input->get('amount');
		$default = $input->get('default');
		$currency = $input->get('currency');
		$returnUrl = $input->get('returnUrl');
		if (!empty($returnUrl)) {
			$returnUrl = base64_decode($returnUrl);
		}
		$placeholder = $input->get('placeholder');

		$button = $this->getButton($description, $amount, $currency, $default, $placeholder, $returnUrl);
		echo '<html><head><meta charset="utf-8"></head><body>'.$button.'</body>';
		// https://fasterjoomla.test/?action=stripebutton_render&amount=720&description=pinolino&currency=EUR
	}

	public function getToolbarData() {
		// $stripePlugin = JPluginHelper::getPlugin('payment','stripe');
		// require_once JPATH_SITE . '/plugins/payment/stripe/helper.php';
		// echo plgPaymentStripeHelper::getToolbarDataJSON($this->stripePlugin);
		// JPluginHelper::importPlugin( 'payment' , 'stripe');
		$json = JEventDispatcher::getInstance($this->stripePlugin)->trigger('getToolbarDataJSON', []);
		if(is_array($json)&&(count($json)>0)){
			echo $json[0];
		} else {
			echo $json;
		}
	}

	public function handlePaymentConfirmComponent() {
		// JPluginHelper::importPlugin( 'payment' , 'stripe');
		JEventDispatcher::getInstance($this->stripePlugin)->trigger('handlePaymentConfirmComponent1', []);
	}


	/**
	 * Parse a json command that looks like:
	 * {stripe-button amount="12.25" currency="EUR" description="buy you a beer"}
	 * (which generates the button)
	 * or - If you do not specify an amount, , I will draw one myself.
	 
	 * or (possible idea for the future):
	 * {stripe-button currency="EUR" source="#yourInputId" description="buy you a beer"}
	 * (without the amount) that binds to the <input id="yourInputId" value="10.00"> and will invoke
	 * the button creation on leaving the field (i.e. clicking the fake button you can specify optionally:
	 * {stripe-button currency="EUR" source="#yourInputId" button="#yourButtonId" description="buy you a beer"}
	 * Note: requires jQuery.
	 *
	 * @param [type] $matches
	 * @return void
	 */
	public function buildButtonCallback($matches) {
		$button = false;
		$original = $matches[0];
		$values = new \stdClass();

		if (count($matches)>1) {
			$match1 = trim($matches[1]);

			foreach(['amount','currency','description','source','button','default','placeholder','returnUrl'] as $attr) {
				if (preg_match('/'.$attr.' *= *"([^\"]+)"/', $match1, $matches)) {
					$values->{$attr} = $matches[1];
				}		
			}

			if (!isset($values->amount)) {
				$outputId = 'stripeButton-'. \rand(100000,980001);
				$url = JUri::current();
				//$url = JUri::base(false). sprintf('index.php?');
				if (strpos($url, '?')) {
					$url.='&';
				} else {
					$url .= '?';
				}
				$url .= sprintf('action=stripebutton_render&description=%s&currency=%s&default=%s&placeholder=%s',
							$values->description, $values->currency, $values->default, $values->placeholder);
				if (!empty($values->returnUrl)) {
						$url .=	"&returnUrl=". base64_encode($values->returnUrl);
						}
							
				$button = sprintf('<div class="stripe-button-container">
					<input type="text" id="%1$s" data-description="" value="%5$s" placeholder="%6$s">
					<button id="%1$s-pay">%3$s</button>

					<script>
						jQuery(function() {
							jQuery("#%1$s-pay").click(function() {
								var amount = jQuery("#%1$s").val();
								if (isNaN(amount)) {
									alert("%4$s");
								} else {
									var url="%2$s&amount="+amount;
									jQuery.get(url, function(data,err) {
										jQuery("#%1$s-output").html(data);
									});
								}
							});
						});
					</script>
					<div id="%1$s-output"></div>
					</div>',
					
					$outputId,
					$url,
					JText::_("STRIPE_BUTTON_CLICK_TO_PAY_AJAX"),
					JText::_("STRIPE_BUTTON_ALERT_NOT_A_NUMBER"),
					$values->default,
					$values->placeholder
					);
			} else {
				if (isset($values->description) && isset($values->amount) && isset($values->currency)) {
					$button = $this->getButton($values->description, $values->amount, $values->currency, @$values->default, @$values->placeholder, @$values->returnUrl);
				} else {
					$button = 'Error in shortcode: No description provided';
				}
			}
		}
		
		if ($button) {
			return $button;
		} else {
			return $original.'<div class="error">ERROR getting Stripe button</div>';}
	}

	private function getButton($description, $amount, $currency, $default='', $placeholder='', $returnUrl='') {
		$vars = new \stdClass();
		$vars->adaptiveReceiverList = null;
		$vars->item_name = $description;
		$currentUrl = JUri::current();
		if (empty($returnUrl)) {
			$returnUrl = $currentUrl;
		}  

		$vars->buttonId = 1+$this->buttonIdCount; 		
		$newUrl = JUri::base(false).'index.php?action=stripebutton_confirm&button_id='.$vars->buttonId
		. '&returnurl='.base64_encode( $returnUrl). '&currentUrl='.base64_encode( $currentUrl);

		$this->buttonIdCount += 1;
		$vars->submiturl = $newUrl;
		$vars->sourceurl = JUri::current();
		$vars->amount = 1*$amount;
		$vars->currency_code = $currency;
		$vars->user_email = null;
		$vars->default = $default;
		$vars->placeholder = $placeholder;
		$vars->returnUrl = $returnUrl;
	
		// $vars->adaptiveReceiverList must be populated if I want stripe_connect to work:2019/05

		$session = JFactory::getSession();
		
		$session->set('stripebutton_vars'.$vars->buttonId,$vars);

		JPluginHelper::importPlugin( 'payment' , 'stripe');
		$html = JEventDispatcher::getInstance($this->stripePlugin)->trigger('onTP_GetHTML', [$vars]);
		if(is_array($html)&&(count($html)>0)){
			return $html[0];
		}
		return false;
	}

	public function loadLanguageStripe($extension = 'plg_payment_stripe', $basePath = JPATH_SITE)
	{
		$lang = JFactory::getLanguage();
		return $lang->load($extension, JPATH_PLUGINS . '/payment/stripe', null, false, true);
	}
	
	public function loadLanguage($extension = '', $basePath = JPATH_SITE)
	{
		if (empty($extension)) {
			$extension = 'plg_' . $this->_type . '_' . $this->_name;
		}
		$lang = JFactory::getLanguage();
		return $lang->load(strtolower($extension), JPATH_PLUGINS . '/' . $this->_type . '/' . $this->_name, null, false, true);
	}
	
}
