<?php
/**
 * Faster Translate creates a drop-down menu in selected views of the Joomla administrator
 * to allow quick access to the translation of the selected elements in Falang and possibly JoomFish. 
 * 
 * @package    Faster Translate
 * @author     Riccardo Zorn {@link http://www.fasterjoomla.com}
 * @created    11-Feb-2013
 * @license    GNU/GPL v2
 */

// No direct access.
defined('_JEXEC') or die;

abstract class modFasterTranslateHelper
{
	/**
	 * Param of the falang component, whether to include the default language in the 
	 * select lists or not. 
	 * @var bool
	 */
	private static $showDefaultLanguage;
	
	/**
	 * Full array of languages, respects the showDefaultLanguage
	 * @var array
	 */
	private static $languages;
	
	/**
	 * falang catid, loose correnspondence to the component name.
	 * @var string
	 */
	private static $catid;
	
	/**
	 * The content (i.e. module, menu etc.) id (only when listView is false)
	 * @var int
	 */
	private static $contentid;
	
	/**
	 * Boolean, true for lists (table views), false for single item edit;
	 * determines the function name;
	 * @var bool
	 */
	private static $listView;
	
	/**
	 * Just in case we're hitting this twice...
	 * @var bool
	 */
	private static $initialised = false;
	
	/**
	 * Just the params of the plugin
	 * @var JRegistry
	 */
	private static $params;
	
	
	/**
	 * Initialise and load language and component data
	 * when false is returned, no menu is rendered.
	 */
	private static function init() {
		if (self::$initialised) { 
			return true;
		}
		if (!file_exists(JPATH_ADMINISTRATOR.'/components/com_falang/falang.xml')) {
			JError::raiseWarning(200,"Faster Translate requires Falang");
			return false;
		}
		
		$params = JComponentHelper::getParams('com_falang');
		$show = $params->get('showDefaultLanguageAdmin',false);
		self::$showDefaultLanguage = $show?true:false;
		
		$result = self::initFalangVariables();
		
		if ($result) {
			self::$languages = self::loadLanguages();
			if (!self::$listView) {
				self::loadTranslationStatus();
			}
		} 
		
		self::$initialised = true;		
		return $result && (count(self::$languages)>0);
	}
	
	/**
	 * Adds some information to the $languages array for item-edit views (not listViews):
	 * Translation date, original content date, translation availability, translation published, 
	 * language of the original item;
	 */
	private static function loadTranslationStatus() {
		$languageIds = join(",",array_keys(self::$languages));
		/*
		 * Falang table:  
		 * id 	language_id 	reference_id 	reference_table 	reference_field 	value 	original_value 	original_text 	modified 	modified_by 	published 
		 *  
		 */
		$db = JFactory::getDbo();
		$query = $db->getQuery(true);
		$query->select(array('modified','published','language_id'))->from('#__falang_content')->
			where('reference_id='.$db->quote(self::$contentid))->
			where('reference_table='.$db->quote(self::$catid))->group(array('modified','published','language_id'));
		$falang = $db->setQuery($query)->loadObjectList('language_id');

		if (!($falang) || count($falang)==0) {
			foreach (self::$languages as $language) {
				$language->translated = false;
			}
			
			return;
		}
		$content = self::loadContent();
		if (!$content) {
			JError::raiseWarning(500, 'Cannot load content info '. $query . ':'.$db->getErrorMsg());
		}
		
		// now add some info to the languages:
		foreach (self::$languages as $language) {
			$language->contentLanguage = $content->language;
			$language->translated = (array_key_exists($language->lang_id,$falang));
			$language->contentPublished = $content->published;
			if ($language->translated) {
				$falangLang = $falang[$language->lang_id];
				if ($content->modified=='n/a') {
					// menu, modules don't have a modified date:
					$language->uptodate = -1; // we don't know yet!
				} else {
					$language->uptodate = ($content->modified <= $falangLang->modified)?'1':'0';
				}
				$language->translationPublished = $falangLang->published;
				
			}
		}
		
	}
	
	/**
	 * 
	 */
	private static function loadContent() {
		$db = JFactory::getDbo();
		$query = $db->getQuery(true);
		
		$query->select('language');
		
		if (self::$catid=='menu' || self::$catid=='modules') {
			// 			menu // published OCCHIO non c'è modified!!!
			// 			modules published OCCHIO non c'è modified!!!
			$query->select("'n/a' as modified");
			
		} else if (self::$catid=='categories') {
			// 			categories: modified_time
			$query->select("modified_time as modified");
			
		} else {
			$query->select('modified');
		}
		
		// convert state to published where appropriate:
		switch (self::$catid) {
			case 'content':
			case 'banners':
			case 'weblinks':
				$query->select('state as published');
				break;
				
			default: // contact_details, categories, menu, modules:
				$query->select('published');
				break;
				
		}
		// now all records have the modified, published, language columns.
		
		$query->from('#__'.self::$catid)->where('id='.$db->quote(self::$contentid));
		
		return $db->setQuery($query)->loadObject();
	}
	
	/**
	 * Return the HTML markup for the buttons as unordered list.
	 * @return unknown
	 */
	public static function getButtons($params)
	{
		self::$params = $params; 
		if (!self::init()) {
			return false;
		}
		self::loadScripts();
		
		return self::renderItems();
	}
		
	/**
	 * Create the two javascript functions, for lists and views, which will ultimately invoke
	 * the falang editing urls:
	 * /administrator/index.php?option=com_falang&task=translate.edit&cid[]=|81|2&catid=content
	 * /administrator/index.php?option=com_falang&task=translate.edit&cid[]=|127|2&catid=modules
	 * 
	 * Note: falang passes the vars with POST, and - if present - it adds as a first item
	 * in the triplet its own translation id: falangid|contentid|languageid;
	 * however the falangid is not mandatory.
	 * This saves us from making extra calls and a dedicated component.
	 */
	private static function loadScripts() {
		$input = JFactory::getApplication()->input;
		$option = $input->get('option');

		
		$editUrl = 'index.php?option=com_falang&task=translate.edit&catid='.self::$catid;
		$script = sprintf('
				// if an item is selected, load the translation function
				function loadFasterTranslationList(languageId) {
					if (document.adminForm.boxchecked.value != 1) {
						alert("%2$s");
						return false;
					}
				
					var cids = document.adminForm["cid[]"];
					// this is pretty strange; the list can be of different types, and 
					// the selector could return a single item; to keep things simple, 
					// I test if it has a length property.
				
					if (typeof cids.length == "undefined") { 
						cids = new Array(cids);
					}
					for (var i=0;i<cids.length;i++) {
						if (cids[i].checked) {
							loadFasterTranslationItem(languageId, cids[i].value);
							break;
						}
					}
				}
				
				// load the translation for the a specific item 
				function loadFasterTranslationItem(languageId, id) {
				
					var cid = "|"+id+"|"+languageId;
					var url = "%1$s&cid[]="+cid+"&select_language_id="+languageId;
					window.open(url);
				}								
				', $editUrl, JText::_("JLIB_HTML_PLEASE_MAKE_A_SELECTION_FROM_THE_LIST"));
		
		$document =JFactory::getDocument();
		$document->addScriptDeclaration($script); 
		$iconsize=16;
		$style = ' 
				ul#fastertranslate {list-style-type:none;float:right;width:250px;right:0}
				ul#fastertranslate li .lcode {font-size:90%;font-style:oblique}
				#fastertranslate.dropdown-menu > li > a {white-space:normal}
				';
		if (version_compare(JVERSION,'3.0','<')) {
			// J 2.5
			$style .= '
					div#fastertranslatediv a,
					div#fastertranslatediv a:focus,
					div#fastertranslatediv a:hover {outline:none;}
					div#fastertranslatediv #menu li:hover a,
					div#fastertranslatediv #menu li {border:none;}
					div#toolbar-box div.m {position:relative;}
					div#fastertranslatediv {position:absolute;right:0px;}
					div.toolbar-list {padding:0 50px 0 0;}
					';
		} else {
			// J 3+
			$style .= '#toolbar {   float: left;
		    			padding-right: 3%;
		    			width: 97%;}
				div#fastertranslatediv {position:absolute;right:0px;}
			';
		}
		
		$document->addStyleDeclaration($style);
	}
	
	/**
	 * Maps the Joomla components to the falang "catid" and sets an extra variable 
	 * to know if we're in list-table or item mode.
	 * 
	 * This is designed to be used on:
	 * Multiple items:
		 * com_content & view=articles;
		 * com_content & view=featured;
		 * com_menus & view=items;
		 * com_categories & view=items;
		 * com_modules & view=module;
	 * 
	 * Individual items:
		 * com_content & view = article;
		 * com_menus & view = item;
		 * com_categories & view = category
		 * com_modules & view = module
		 * 
	 * falang catid can be:
			banners
			categories
			contact_details
			content
			menu
			modules
			weblinks
	 */
	private static function initFalangVariables() {
		$input = JFactory::getApplication()->input;
		$option = $input->get('option');
		$view = $input->get('view');
		$id = $input->get('id',0);

		
		switch ($option) {
			case 'com_menus':
				$catid = 'menu';
				break;
			case 'com_contact':
				$catid = 'contact_details';
				break;
			default:
				$catid = str_replace('com_', '', $option);
				break;
		}
		
		$listView = true;
		
		switch ($option.'.'.$view) {
			case 'com_banners.banners':
			case 'com_banners.':
			case 'com_categories.categories':
			case 'com_categories.':
			case 'com_contact.contacts':
			case 'com_contact.':
			case 'com_content.articles':
			case 'com_content.featured':
			case 'com_content.':
			case 'com_menus.items':
			case 'com_menus.':
			case 'com_modules.modules':
			case 'com_modules.':
			case 'com_weblinks.weblinks':
			case 'com_weblinks.':
				$listView = true;
				break;	
			case 'com_banners.banner':		
			case 'com_categories.category':
			case 'com_contact.contact':
			case 'com_content.article':
			case 'com_menus.item':
			case 'com_modules.module':
			case 'com_weblinks.weblink':
				$listView = false;
				if ($id == 0) {
					return false;
				}
				break;
			default:
				// we're not in a supported component/view: exit.
				return false;
				break;
		}
		self::$listView = $listView;
		self::$catid = $catid;
		self::$contentid = $id;
		return true;
	}
	
	/**
	 * Renders an array of items as an unordered HTML list (ul - li).
	 * @param unknown $buttons
	 */
	private static  function renderItems() {
		$catid = self::$catid;
		
		if (!self::$listView) {
			$input = JFactory::getApplication()->input;
			$id =  $input->get('id');
		}
		
		$html[] = '';
		
		foreach (self::$languages as $languageId=>$language) {
			if (self::$listView) {
				$function = " onclick='loadFasterTranslationList(\"$languageId\")' ";
			} else {
				$function = " onclick='loadFasterTranslationItem(\"$languageId\", \"$id\")' ";
			}
			$icons = "";
			
			if (!self::$listView) {
				$red = "components/com_falang/assets/images/status_r.png";
				$yellow = "components/com_falang/assets/images/status_y.png";
				$green = "components/com_falang/assets/images/status_g.png";
				$notpub = "modules/mod_fastertranslate/assets/images/notpublished.png";
				$dontknow = "components/com_falang/assets/images/status_y.png";
				
				if ($language->translated) {
					if ($language->uptodate==1)
						$icons = sprintf("<img src='$green' alt='ok' title='%s'/>",JText::_("MOD_FASTERTRANSLATE_TRANSLATION_UPTODATE"));
					else if ($language->uptodate==0)
						$icons = sprintf("<img src='$yellow' alt='old' title='%s'/>",JText::_("MOD_FASTERTRANSLATE_TRANSLATION_OLD"));
					else if ($language->uptodate==-1)
						$icons = sprintf("<img src='$dontknow' alt='old' title=\"%s\"/>",JText::_("MOD_FASTERTRANSLATE_TRANSLATION_DONTKNOW"));
					if (!$language->translationPublished) {
						$icons .= sprintf("<img src='$notpub' alt='n/p' title='%s'/>",JText::_("MOD_FASTERTRANSLATE_TRANSLATION_NOTPUBLISHED"));
					}
				} else {
					$icons = sprintf("<img src='$red' alt='n/a' title='%s'/>",JText::_("MOD_FASTERTRANSLATE_TRANSLATION_NA"));
				}
			}
			
			$html[] = sprintf('<li>
					<a href="#" %s>
					<img src="%s"/>
					%s					
					<span class="lcode">
						%s
					</span>
					<span class="pull-right">
						%s
					</span>
					</a>
					
					</li>',
					$function,
					$language->image,
					$language->title,
					$language->lang_code,
					$icons
					
					
				);
		}
		
		$html[] = self::renderHelp();
		
		$html[] = '';
		return join("\n",$html);
	}
	
	private static function renderHelp() {
		$help = self::$params->get('showhelp', '1')=='1';
		if ($help) {
			$link = 'http://www.fasterjoomla.com/en/extensions/faster-translate-quick-access-to-falang-translations?tmpl=component';
			JHtml::_('behavior.modal');
			$html[] = sprintf('<li class="divider"><span></span></li>');
			$html[] = sprintf('<li><a href="%s" class="modal" rel="{handler:\'iframe\', size:{x:700,y:400}}">%s</a></li>',
				$link,JText::_("JHELP"));
			return join("",$html);
		}
	}

	/**
	 * Load the Joomla languages as found 
	 * in administrator->extensions->Language Manager->Content.
	 * @return unknown|multitype:
	 */
	private static function loadLanguages() {
		$lang = array();
		$db = JFactory::getDbo();
		$query = $db->getQuery(true);
		$query->select(array('lang_id','title', 'lang_code','image'))->from('#__languages')->
			where('published=1')->order('ordering');
		
		
		
		$db->setQuery($query);
		$results = $db->loadObjectList('lang_id');
		
		if (!self::$showDefaultLanguage) {
			//$query->where('ordering>1');
			
			$params = JComponentHelper::getParams('com_languages');
			$joomlaDefaultLanguage = $params->get("site", 'en-GB');
			
			foreach($results as $key=>$result) {
				if ($result->lang_code == $joomlaDefaultLanguage) {
					unset($results[$key]);
					break;
				}
			}
		}
		
		foreach($results as $result) {
			$result->image = '/media/mod_languages/images/'.strtolower(str_replace('-','_',($result->image))).'.gif';
		}
		
		return $results;
		
	}
}
